package uk.ac.manchester.cs.schneidt.decomp;

import static uk.ac.manchester.cs.schneidt.decomp.DecompositionWorking.*;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jxl.Cell;
import jxl.Sheet;
import jxl.Workbook;
import jxl.read.biff.BiffException;

import org.apache.commons.io.IOUtils;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLDatatype;
import org.semanticweb.owlapi.model.OWLEntity;
import org.semanticweb.owlapi.model.OWLLogicalAxiom;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLOntologyCreationException;
import org.semanticweb.owlapi.model.OWLOntologyStorageException;
import org.semanticweb.owlapi.util.AutoIRIMapper;

import uk.ac.manchester.cs.owlapi.modularity.ModuleType;
import uk.ac.manchester.cs.owlapi.modularity.SyntacticLocalityModuleExtractor;

/**
 * Created by IntelliJ IDEA. User: schneider Date: 07.04.11 Time: 17:08
 */
public class MaxAntichainsForSubsets {
	static String ontologyFileName;
	static String sigFileName;
	static String oldXLSFileName;
	static String newXLSFileName;

	public static void evaluateParameters(String[] args) {
		try {
			ontologyFileName = args[0];
			sigFileName = args[1];
			oldXLSFileName = args[2];
			newXLSFileName = args[3];
		} catch (ArrayIndexOutOfBoundsException e) {
			System.out.println("Please provide four arguments: ");
			System.out.println("  * The name of the ontology to decompose");
			System.out.println("  * The name of the file that contains the signatures for the subontologies");
			System.out.println("  * The name of the file that contains the xls file from the previous experiment");
			System.out.println("  * The name of the file that is to contain the xls generated");
			throw new RuntimeException();
		}
	}

	public static OWLOntology loadOntology() throws OWLOntologyCreationException {
		File file = new File(ontologyFileName);
		AutoIRIMapper autoIRIMapper = new AutoIRIMapper(new File(ontologyFileName), true);
		manager.addIRIMapper(autoIRIMapper);
		return manager.loadOntologyFromOntologyDocument(file);
	}

	public static Set<OWLEntity> getSuitableEntities(OWLOntology ontology) {
		Set<OWLClass> allClss = new HashSet<OWLClass>(ontology.getClassesInSignature(true));
		Set<OWLEntity> allEnts = new HashSet<OWLEntity>(allClss);
		allEnts.addAll(ontology.getObjectPropertiesInSignature(true));
		allEnts.addAll(ontology.getDataPropertiesInSignature(true));
		Set<OWLEntity> goodEnts = new HashSet<OWLEntity>(allEnts);
		goodEnts.remove(factory.getOWLThing());
		for (OWLEntity ent : allEnts) {
			if (OWLDatatype.class.isAssignableFrom(ent.getClass())) {
				goodEnts.remove(ent);
			} else if (ent.toString().indexOf("clusive") >= 0) {
				goodEnts.remove(ent);
			}
		}
		return goodEnts;
	}

	public static List<Set<OWLEntity>> loadSignatures(OWLOntology ont) throws IOException {
		List<Set<OWLEntity>> sigs = new LinkedList<Set<OWLEntity>>();
		BufferedReader file = new BufferedReader(new InputStreamReader(new FileInputStream(
				sigFileName)));
		Set<OWLEntity> currentSig = new HashSet<OWLEntity>();
		while (file.ready()) {
			String line = file.readLine();
			if (line.equals("")) {
				sigs.add(currentSig);
				currentSig = new HashSet<OWLEntity>();
			} else if (!line.equals("{}")) {
				OWLEntity ent = getEntity(line, ont);
				currentSig.add(ent);
			}
		}
		file.close();
		return sigs;
	}

	private static OWLEntity getEntity(String name, OWLOntology ont) {
		IRI iri = IRI.create(name);
		if (ont.containsClassInSignature(iri)) {
			return factory.getOWLClass(iri);
		}
		if (ont.containsObjectPropertyInSignature(iri)) {
			return factory.getOWLObjectProperty(iri);
		}
		if (ont.containsDataPropertyInSignature(iri)) {
			return factory.getOWLDataProperty(iri);
		}
		throw new RuntimeException("There is no class or property of name " + name + ".");
	}

	private static int[][] readPreviousData() throws BiffException, IOException {
		Workbook workbook = Workbook.getWorkbook(new File(oldXLSFileName));
		Sheet sheet = workbook.getSheet("Overview");
		int offsetRows = 5;
		int numberRows = sheet.getRows() - offsetRows;
		int numberColumns = sheet.getColumns();
		int[][] previousData = new int[numberRows][numberColumns];
		for (int i = 0; i < numberRows; i++) {
			for (int j = 0; j < numberColumns; j++) {
				Cell cell = sheet.getCell(j, i + offsetRows);
				previousData[i][j] = Integer.parseInt(cell.getContents());
			}
		}
		return previousData;
	}

	private static int getMaxAntichainLength(OWLOntology ont, Set<OWLEntity> goodEnts,
			Set<OWLEntity> signature, File outputPath, int[] previousRow)
			throws OWLOntologyCreationException, OWLOntologyStorageException, IOException,
			InterruptedException {
		// Generate subontology
		SyntacticLocalityModuleExtractor extractor = new SyntacticLocalityModuleExtractor(manager,
				ont, ModuleType.STAR);
		OWLOntology subont = extractor.extractAsOntology(
				signature,
				IRI.create("http://tmp.de/" + System.currentTimeMillis() + ".owl"));
		Set<OWLEntity> subontSig = new HashSet<OWLEntity>(subont.getSignature());
		subontSig.retainAll(goodEnts);
		System.out.println(subont.getLogicalAxiomCount() + ", " + subontSig.size() + " <-> "
				+ previousRow[1] + ", " + previousRow[2]);
		// Generate decomposition of subontology
		Set<OWLLogicalAxiom> axioms = subont.getLogicalAxioms();
		Set<OWLLogicalAxiom> syntTautologies = new HashSet<OWLLogicalAxiom>(axioms);
		Set<OWLAxiom> mod0 = extractModuleForEmptySig(subont, outputPath.toURI()
				+ "/modules/module0_" + System.currentTimeMillis() + ".owl");
		Map<OWLLogicalAxiom, Set<OWLLogicalAxiom>> genuineModules = extractModules(
				ont,
				axioms,
				syntTautologies,
				outputPath.toURI() + "/modules/");
		removeSyntTautologiesAndGlobalAxiomsFrom(axioms, syntTautologies, mod0);
		Map<OWLLogicalAxiom, Set<OWLLogicalAxiom>> atomOfAxiom = initialiseAtomsOfAxioms(axioms);
		Set<OWLLogicalAxiom> generatingAxioms = getAtoms(axioms, genuineModules, atomOfAxiom);
		Map<OWLLogicalAxiom, Integer> axiomIDHashMap = generateAxiomIDMap(
				atomOfAxiom,
				generatingAxioms,
				outputPath.toURI() + "/atoms/");
		// int cardOfMaxAtom = getMaxSizeOfAtom(atomOfAxiom, generatingAxioms);
		Map<OWLLogicalAxiom, Set<OWLLogicalAxiom>> transClosureOfAtDec = getTransClosureOfAtDec(
				genuineModules,
				generatingAxioms);
		// Map<OWLLogicalAxiom, Set<OWLLogicalAxiom>> axiomsToBeRemoved =
		// getAxiomsToBeRemoved(generatingAxioms, transClosureOfAtDec);
		// Map<OWLLogicalAxiom, Set<OWLLogicalAxiom>> reducedAtDec =
		// getReducedAtDec(generatingAxioms, transClosureOfAtDec,
		// axiomsToBeRemoved);
		// Map<OWLLogicalAxiom, Integer> cardOfAtoms =
		// getCardOfAtoms(atomOfAxiom, generatingAxioms);
		// Set<OWLLogicalAxiom> unrelatedAtoms =
		// getUnrelatedAtoms(generatingAxioms, reducedAtDec);
		String decompFileName = outputPath.toString() + "/decomp.tmp.txt";
		saveDecomposition(generatingAxioms, axiomIDHashMap, transClosureOfAtDec, decompFileName);
		// Run bipart.py on decomposition
		String bipartScriptName = outputPath.toString() + "/bipart.py";
		ProcessBuilder pb = new ProcessBuilder("python", bipartScriptName, decompFileName);
		// Map<String, String> env = pb.environment();
		pb.directory(outputPath);
		Process p = pb.start();
		InputStream inputStream = p.getInputStream();
		File bipartOutputFile = new File(outputPath.toString() + "/decomp.bipart.tmp.txt");
		BufferedWriter out = new BufferedWriter(new FileWriter(bipartOutputFile, false));
		IOUtils.copy(inputStream, out);
		out.close();
		int exitValue = p.waitFor();
		if (exitValue != 0) {
			throw new RuntimeException("Process returned exit value " + exitValue + ".");
		}
		// Get length of maximal antichain
		// TODO: Continue
		// Remove all temporary ontologies
		manager.removeOntology(subont);
		return 0;
	}

	private static int[][] processSubontologies(OWLOntology ont, Set<OWLEntity> goodEnts,
			List<Set<OWLEntity>> signatures, int[][] previousData, File outputPath)
			throws OWLOntologyCreationException, OWLOntologyStorageException, IOException,
			InterruptedException {
		List<int[]> newData = new LinkedList<int[]>();
		for (int i = 0; i < previousData.length; i++) {
			int[] previousRow = previousData[i];
			if (previousRow[3] == 0) {
				break; // First row of zeros has been reached
			}
			int[] newRow = new int[5];
			System.arraycopy(previousRow, 0, newRow, 0, 4);
			newRow[4] = getMaxAntichainLength(
					ont,
					goodEnts,
					signatures.get(i),
					outputPath,
					previousRow);
			newData.add(newRow);
		}
		return newData.toArray(new int[newData.size()][]);
	}

	public static void main(String[] args) {
		try {
			// Evaluate Parameters
			evaluateParameters(args);
			// Load ontology
			System.out.println("LOADING ONTOLOGY  " + ontologyFileName);
			OWLOntology ont = loadOntology();
			Set<OWLEntity> goodEnts = getSuitableEntities(ont);
			// Load signature file
			System.out.println("READING SIGNATURES FROM  " + sigFileName);
			List<Set<OWLEntity>> signatures = loadSignatures(ont);
			// Load "old" xls table
			System.out.println("READING PREVIOUS DATA FROM  " + oldXLSFileName);
			int[][] previousData = readPreviousData();
			// Process subsets
			System.out.print("PROCESSING SUBONTOLOGIES");
			File outputPath = new File(ontologyFileName).getParentFile();
			int[][] newData = processSubontologies(
					ont,
					goodEnts,
					signatures,
					previousData,
					outputPath);
			System.out.println();
			// Write to "new" xls table
			// TODO
		} catch (OWLOntologyCreationException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (BiffException e) {
			e.printStackTrace();
		} catch (OWLOntologyStorageException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}
}
